# -*- coding: utf-8 -*-
import logging
import sqlite3
import os
import time
import subprocess
import textwrap
import math
import sys
import select
from datetime import datetime
from escpos.printer import Serial
from escpos.capabilities import Profile
from evdev import InputDevice, categorize, ecodes
from RPLCD.i2c import CharLCD
from gpiozero import Button

# Konfiguration laden
import config
from ftp_download import download_mitglieder
from ftp_upload import upload_backup

# --- LOGGING ---
logging.basicConfig(filename=config.LOG_FILE, level=logging.INFO, format='%(asctime)s - %(message)s')

# --- INITIALISIERUNG ---
try:
    lcd = CharLCD('PCF8574', config.LCD_ADDRESS)
    lcd.create_char(0, (0x0A, 0x00, 0x0E, 0x01, 0x0F, 0x11, 0x0F, 0x00)) # ä
    lcd.create_char(1, (0x0A, 0x00, 0x0E, 0x11, 0x11, 0x11, 0x0E, 0x00)) # ö
    lcd.create_char(2, (0x0A, 0x00, 0x11, 0x11, 0x11, 0x13, 0x0D, 0x00)) # ü
    lcd.create_char(3, (0x00, 0x1E, 0x11, 0x1E, 0x11, 0x11, 0x1E, 0x10)) # ß
except:
    lcd = None

btn_ja = Button(config.PIN_BUTTON_JA, pull_up=True, bounce_time=0.3)
btn_nein = Button(config.PIN_BUTTON_NEIN, pull_up=True, bounce_time=0.3)

last_minute = -1

# --- HILFSFUNKTIONEN ---

def fix_umlaute(text):
    if not text: return ""
    return text.replace('ä', '\x00').replace('Ä', '\x00')\
               .replace('ö', '\x01').replace('Ö', '\x01')\
               .replace('ü', '\x02').replace('Ü', '\x02')\
               .replace('ß', '\x03')

def update_lcd(line1=None, line2="Karte scannen..."):
    global last_minute
    if not lcd: return
    try:
        now = datetime.now()
        lcd.clear()
        if line1 is None:
            line1 = now.strftime("%d.%m.%y   %H:%M")
            last_minute = now.minute
            lcd.write_string(fix_umlaute(line1[:16]))
        else:
            lcd.write_string(fix_umlaute(line1[:16]).center(16))
            
        if line2:
            lcd.cursor_pos = (1, 0)
            lcd.write_string(fix_umlaute(line2[:16]).center(16))
    except: pass

def get_user_info(card_id):
    try:
        conn = sqlite3.connect(config.DB_MITGLIEDER)
        c = conn.cursor()
        c.execute("SELECT vorname, name, gartennummer FROM users WHERE card_id = ?", (card_id,))
        res = c.fetchone()
        conn.close()
        if res: return {"vorname": res[0], "name": res[1], "garten": res[2]}
    except: pass
    return {"vorname": "Unbekannt", "name": card_id, "garten": "-"}

def wait_for_button():
    time.sleep(0.5)
    while True:
        if btn_ja.is_pressed:
            return True
        if btn_nein.is_pressed:
            return False
        time.sleep(0.1)

# --- DRUCKER LOGIK ---

def setup_rfcomm():
    try:
        subprocess.run(["sudo", "rfcomm", "release", "0"], capture_output=True)
        time.sleep(0.5)
        subprocess.run(["sudo", "rfcomm", "bind", "0", config.PRINTER_MAC], capture_output=True)
        time.sleep(2.0)
        return os.path.exists(config.PRINTER_PORT)
    except: return False

def print_receipt(user, start_t, end_t, duration, gewertet):
    if not getattr(config, 'ENABLE_PRINTING', False): return
    if not os.path.exists(config.PRINTER_PORT):
        if not setup_rfcomm(): return
    p = None
    try:
        profile = Profile()
        profile.profile_data['media']['width']['pixel'] = 384
        p = Serial(config.PRINTER_PORT, baudrate=9600, profile=profile, timeout=5)
        if os.path.exists(config.LOGO_PATH):
            p.set(align='center')
            p.image(config.LOGO_PATH)
        
        p.set(align='center', bold=True)
        p.text("\nVerein für Garten- und\n")
        p.text("Naturfreunde\n \"Nach Feierabend\" e.V.\n")
        p.text("Weissensee/Thueringen\n")
        p.text("-" * 32 + "\n")
        
        p.set(align='left', bold=False)
        p.text(f"Name:   {user['vorname']} {user['name']}\n")
        p.text(f"Garten: Nr. {user['garten']}\n")
        p.text(f"Von:    {start_t}\n")
        p.text(f"Bis:    {end_t}\n")
        p.text("-" * 32 + "\n")
        
        p.set(bold=True)
        p.text(f"Dauer:    {duration} h\n")
        p.text(f"Wertung:  {gewertet} Std\n")
        
        p.set(align='center', bold=False)
        p.text("\nDanke fuer deine Arbeit!\n")
        p.text("Wir wuenschen dir noch einen\n")
        p.text("schoenen Tag! Der Vorstand\n\n")
        
        p.set(bold=True)
        p.text("Bewahre diesen Beleg gut auf!\n")
        p.text("Falls etwas mit der Abrechnung\n")
        p.text("fehlerhaft sein sollte.\n")
        
        p.set(align='center', bold=False)
        p.cut()
    except Exception as e:
        logging.error(f"Druckfehler: {e}")
    finally:
        if p: p.close()

def print_summary(regulaer, zwang):
    if not getattr(config, 'ENABLE_PRINTING', False): return
    if not os.path.exists(config.PRINTER_PORT):
        if not setup_rfcomm(): return
    p = None
    try:
        profile = Profile()
        p = Serial(config.PRINTER_PORT, baudrate=9600, profile=profile, timeout=5)
        
        if os.path.exists(config.LOGO_PATH):
            p.set(align='center')
            p.image(config.LOGO_PATH)
        
        p.set(align='center', bold=True)
        p.text("\nVerein für Garten- und\n")
        p.text("Naturfreunde\n \"Nach Feierabend\" e.V.\n")
        p.text("Weissensee/Thueringen\n")
        p.text("-" * 32 + "\n")
        p.text("TAGESZUSAMMENFASSUNG\n")
        p.text(datetime.now().strftime("%d.%m.%Y") + "\n")
        p.text("-" * 32 + "\n")
        
        p.set(align='left', bold=True)
        p.text("REGULAER BEENDET:\n")
        p.set(bold=False)
        for r in regulaer:
            p.text(f"{r[0]}\n")
            p.text(f" Garten: Nr. {r[6]}\n")
            p.text(f" {r[1][-5:]}-{r[2][-5:]} ({r[4]} Std)\n")
        
        if zwang:
            p.text("-" * 32 + "\n")
            p.set(bold=True)
            p.text("ZWANGS-LOGOUT (Wertung 0):\n")
            p.set(bold=False)
            for z in zwang:
                # Da zwang_liste nur Name und Startzeit enthält, holen wir die Gartennummer über die Datenbank oder Card_ID
                p.text(f"{z[0]}\n")
                # Wir zeigen hier die Startzeit an
                p.text(f" Start: {z[1][-5:]}\n")
        
        p.text("-" * 32 + "\n")
        p.cut()
    except Exception as e:
        logging.error(f"Zusammenfassung Druckfehler: {e}")
    finally:
        if p: p.close()

# --- HAUPTLOGIK ---

def safe_upload():
    """Hilfsfunktion für LCD Info bei Upload"""
    update_lcd("DATEN-UPLOAD", "Bitte warten...")
    try:
        if upload_backup():
            return True
        else:
            update_lcd("UPLOAD FEHLER", "Pruefe Internet")
            time.sleep(3.0)
            return False
    except Exception as e:
        logging.error(f"Manueller Upload Fehler: {e}")
        update_lcd("FEHLER", "Upload fehlgesch.")
        time.sleep(3.0)
        return False

def process_card(card_id):
    global last_minute
    WAIT_STATS = 3.0
    WAIT_NAME = 2.0
    try:
        conn_anw = sqlite3.connect(config.DB_ANWESENHEIT)
        c = conn_anw.cursor()
        c.execute("CREATE TABLE IF NOT EXISTS scans (card_id TEXT, start_time TEXT, end_time TEXT, duration TEXT, gewertet TEXT, user_name TEXT, gartennummer TEXT)")
        
        user = get_user_info(card_id)
        full_name = f"{user['vorname']} {user['name']}"
        garten_nr = user['garten']

        c.execute("SELECT rowid, start_time FROM scans WHERE card_id = ? AND end_time IS NULL", (card_id,))
        open_scan = c.fetchone()

        if not open_scan:
            c.execute("SELECT user_name FROM scans WHERE gartennummer = ? AND end_time IS NULL", (garten_nr,))
            andere_karte = c.fetchone()
            if andere_karte:
                update_lcd("FEHLER GARTEN", "BEREITS GESCANNT")
                time.sleep(2.0)
                last_minute = -1
                update_lcd()
                conn_anw.close()
                return

        if open_scan:
            rowid, start_str = open_scan
            now_dt = datetime.now()
            now_str = now_dt.strftime("%d.%m.%Y %H:%M")
            start_dt = datetime.strptime(start_str, "%d.%m.%Y %H:%M")
            diff = now_dt - start_dt
            total_minutes = diff.total_seconds() / 60
            duration_str = f"{int(total_minutes // 60)}:{int(total_minutes % 60):02d}"
            
            gewertet = 0
            if config.ENABLE_WERTUNG:
                if total_minutes >= config.MINUTEN_STUFE_2: gewertet = config.WERTUNG_STUFE_2
                elif total_minutes >= config.MINUTEN_STUFE_1: gewertet = config.WERTUNG_STUFE_1
            else:
                gewertet = math.ceil(total_minutes / 60)

            update_lcd(f"Zeit: {duration_str} h", f"Wertung: {gewertet} Std")
            time.sleep(WAIT_STATS)

            update_lcd("Jetzt ausloggen?", "grün=JA rot=NEIN")
            if not wait_for_button():
                update_lcd("ABGEBROCHEN", "Scannen bereit")
                time.sleep(2.0)
                conn_anw.close()
                return

            druck_erfolgt = False
            if getattr(config, 'ENABLE_PRINTING', False):
                update_lcd("Drucke Beleg...", "Bitte warten")
                print_receipt(user, start_str, now_str, duration_str, gewertet)
                druck_erfolgt = True
            
            save_allowed = True
            if druck_erfolgt:
                update_lcd("Beleg korrekt?", "grün=JA rot=NEIN")
                save_allowed = wait_for_button()

            if save_allowed:
                c.execute("UPDATE scans SET end_time=?, duration=?, user_name=?, gartennummer=?, gewertet=? WHERE rowid=?",
                          (now_str, duration_str, full_name, garten_nr, str(gewertet), rowid))
                conn_anw.commit()
                if lcd:
                    parts = textwrap.wrap(full_name, 16, break_long_words=False)
                    for part in parts:
                        lcd.clear()
                        lcd.write_string("Auf Wiedersehen".center(16))
                        lcd.cursor_pos = (1, 0)
                        lcd.write_string(fix_umlaute(part).center(16))
                        time.sleep(WAIT_NAME)
            else:
                update_lcd("NICHT GESPEICHERT", "Scannen bereit")
                time.sleep(2.0)
        else:
            now_str = datetime.now().strftime("%d.%m.%Y %H:%M")
            c.execute("INSERT INTO scans (card_id, start_time, user_name, gartennummer) VALUES (?, ?, ?, ?)",
                      (card_id, now_str, full_name, garten_nr))
            conn_anw.commit()
            if lcd:
                parts = textwrap.wrap(full_name, 16, break_long_words=False)
                for part in parts:
                    lcd.clear()
                    lcd.write_string("Willkommen".center(16))
                    lcd.cursor_pos = (1, 0)
                    lcd.write_string(fix_umlaute(part).center(16))
                    time.sleep(WAIT_NAME)
        conn_anw.close()
    except Exception as e:
        logging.error(f"Fehler: {e}")
        update_lcd("FEHLER", "DB PRÜFEN")
        time.sleep(2)
    last_minute = -1
    update_lcd()

def main():
    global last_minute
    if lcd: update_lcd("DATEN-ABGLEICH", "Bitte warten...")
    try:
        download_mitglieder()
        if lcd: update_lcd("UPDATE OK", "Daten geladen"); time.sleep(1.0)
    except: pass

    device = None
    for i in range(15):
        if os.path.exists(config.RFID_PATH):
            try:
                device = InputDevice(config.RFID_PATH); device.grab(); break
            except: pass
        time.sleep(1)

    if not device:
        update_lcd("RFID FEHLT", "Kabel prüfen")
        return

    update_lcd()
    time.sleep(1)

    try:
        id_buffer = ""
        while True:
            if btn_ja.is_pressed and btn_nein.is_pressed:
                start_p = time.time()
                while btn_ja.is_pressed and btn_nein.is_pressed:
                    if time.time() - start_p >= 2.0:
                        try:
                            conn = sqlite3.connect(config.DB_ANWESENHEIT)
                            c = conn.cursor()
                            c.execute("SELECT rowid, user_name, start_time FROM scans WHERE end_time IS NULL")
                            aktive = c.fetchall()
                            
                            zwang_liste = []
                            gehe_zu_druck = False

                            if aktive:
                                update_lcd(f"{len(aktive)} AKTIVE", "Alle ausloggen?")
                                if wait_for_button():
                                    now_str = datetime.now().strftime("%d.%m.%Y %H:%M")
                                    for rowid, name, start_str in aktive:
                                        c.execute("UPDATE scans SET end_time=?, duration='0:00', gewertet='0' WHERE rowid=?", (now_str, rowid))
                                        zwang_liste.append((name, start_str))
                                    conn.commit()
                                    gehe_zu_druck = True
                                else:
                                    conn.close()
                                    update_lcd()
                                    last_minute = -1
                                    break 
                            else:
                                update_lcd("KEINE AKTIVEN", "Report drucken?")
                                if wait_for_button():
                                    gehe_zu_druck = True
                                else:
                                    conn.close()
                                    safe_upload()
                                    update_lcd("FERTIG", "Menu wird geladen")
                                    time.sleep(2.0)
                                    device.ungrab(); btn_ja.close(); btn_nein.close()
                                    os.execv(sys.executable, ['python3', os.path.join(config.BASE_DIR, "main_menu.py")])
                                    return

                            if gehe_zu_druck:
                                if getattr(config, 'ENABLE_PRINTING', False):
                                    update_lcd("DRUCKE REPORT", "Bitte warten")
                                    heute_str = datetime.now().strftime("%d.%m.%Y")
                                    c.execute("SELECT user_name, start_time, end_time, duration, gewertet, card_id, gartennummer FROM scans WHERE end_time IS NOT NULL")
                                    alle_heute = c.fetchall()
                                    
                                    regulaer = []
                                    for r in alle_heute:
                                        if r[1].startswith(heute_str):
                                            ist_zwang = any(z[0] == r[0] and z[1] == r[1] for z in zwang_liste)
                                            if not ist_zwang:
                                                regulaer.append(r)
                                    
                                    print_summary(regulaer, zwang_liste)
                                    
                                    update_lcd("Druck OK?", "grün=JA rot=NEIN")
                                    if wait_for_button():
                                        conn.close()
                                        safe_upload()
                                        update_lcd("FERTIG", "Menu wird geladen")
                                        time.sleep(2.0)
                                        device.ungrab(); btn_ja.close(); btn_nein.close()
                                        os.execv(sys.executable, ['python3', os.path.join(config.BASE_DIR, "main_menu.py")])
                                        return
                                    else:
                                        conn.close()
                                        update_lcd("ABGEBROCHEN", "Scannen bereit")
                                        time.sleep(2.0)
                                        last_minute = -1
                                        update_lcd()
                                        break
                                else:
                                    conn.close()
                                    safe_upload()
                                    update_lcd("FERTIG", "Menu wird geladen")
                                    time.sleep(2.0)
                                    device.ungrab(); btn_ja.close(); btn_nein.close()
                                    os.execv(sys.executable, ['python3', os.path.join(config.BASE_DIR, "main_menu.py")])
                                    return
                            
                        except Exception as e:
                            logging.error(f"Fehler Feierabend: {e}")
                            try: conn.close()
                            except: pass
                            update_lcd("FEHLER", "ABBRUCH")
                            time.sleep(2.0)

                        update_lcd()
                    time.sleep(0.1)

            now = datetime.now()
            if now.minute != last_minute and not id_buffer:
                update_lcd()

            r, w, x = select.select([device.fd], [], [], 0.1)
            if r:
                for event in device.read():
                    if event.type == ecodes.EV_KEY and categorize(event).keystate == 1:
                        key_code = categorize(event).keycode
                        if key_code == 'KEY_ENTER':
                            if id_buffer:
                                process_card(id_buffer)
                                id_buffer = ""
                        else:
                            val = key_code.replace('KEY_', '')
                            if len(val) == 1 and val.isdigit():
                                id_buffer += val
            time.sleep(0.05)
    except KeyboardInterrupt: pass
    finally:
        if device: device.ungrab()
        btn_ja.close(); btn_nein.close()

if __name__ == "__main__":
    main()