# -*- coding: utf-8 -*-
import os
import time
import sys
import subprocess
import socket
import re
from datetime import datetime
from RPLCD.i2c import CharLCD
from gpiozero import Button

# Konfiguration laden
import config

# --- INITIALISIERUNG ---
try:
    lcd = CharLCD('PCF8574', config.LCD_ADDRESS)
except:
    lcd = None

# Taster für Navigation (Rot = NEIN, Grün = JA)
btn_ja = Button(config.PIN_BUTTON_JA, pull_up=True, bounce_time=0.3)
btn_nein = Button(config.PIN_BUTTON_NEIN, pull_up=True, bounce_time=0.3)

# --- SYSTEM-DATEN FUNKTIONEN ---

def get_version_info():
    try:
        version = getattr(config, 'VERSION', "v1.0")
        return f"{version}"
    except:
        return "Version fehlerhaft"

def get_ip_address():
    try:
        s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        s.connect(("8.8.8.8", 80))
        ip = s.getsockname()[0]
        s.close()
        return ip
    except:
        return "Kein Netz"

def get_wifi_name():
    try:
        ssid = subprocess.check_output(['iwgetid', '-r']).decode('utf-8').strip()
        return ssid if ssid else "Nicht verbunden"
    except:
        return "Kein WLAN"

def get_wifi_signal():
    """Liest die WLAN-Signalstärke aus /proc/net/wireless"""
    try:
        cmd = "iwconfig wlan0 | grep -i 'Link Quality'"
        output = subprocess.check_output(cmd, shell=True).decode('utf-8')
        # Extrahiert z.B. "70/70"
        quality = re.search(r'Link Quality=(\d+/\d+)', output).group(1)
        # Extrahiert den dBm Wert, falls gewünscht
        level = re.search(r'Signal level=(-\d+ dBm)', output).group(1)
        
        # Umrechnung in Prozent
        q_parts = quality.split('/')
        percent = int((int(q_parts[0]) / int(q_parts[1])) * 100)
        
        return f"{percent}% ({level})"
    except:
        return "N/A"

def get_sd_status():
    """Prüft die Gesundheit durch einen echten Schreibtest auf der Daten-Partition"""
    try:
        test_file = os.path.join(os.path.dirname(config.DB_ANWESENHEIT), ".sd_test")
        try:
            with open(test_file, "w") as f:
                f.write("OK")
            os.remove(test_file)
            return "Status: GESUND"
        except:
            return "Status: SCHREIBGESCH."
    except:
        return "Status: UNBEKANNT"

def get_cpu_temp():
    try:
        res = os.popen('vcgencmd measure_temp').readline()
        return res.replace("temp=", "").replace("'C\n", " C")
    except:
        return "0.0 C"

def get_uptime():
    try:
        raw_uptime = subprocess.check_output(['uptime', '-p']).decode('utf-8')
        uptime_str = raw_uptime.replace('up ', '').strip()
        hours = 0
        minutes = 0
        parts = uptime_str.split(',')
        for part in parts:
            if 'hour' in part:
                hours = [int(s) for s in part.split() if s.isdigit()][0]
            elif 'minute' in part:
                minutes = [int(s) for s in part.split() if s.isdigit()][0]
        return f"{hours}h {minutes}min"
    except:
        return "0h 0min"

def get_disk_space():
    try:
        stat = os.statvfs('/')
        free = (stat.f_bavail * stat.f_frsize) / (1024**3)
        return f"Frei: {free:.1f} GB"
    except:
        return "Fehler"

# --- ANZEIGE-LOGIK ---

def main():
    if not lcd:
        return

    pages = [
        ("VERSION INFO", get_version_info),
        ("IP ADRESSE", get_ip_address),
        ("WLAN NAME", get_wifi_name),
        ("WLAN SIGNAL", get_wifi_signal),
        ("SD-GESUNDHEIT", get_sd_status),
        ("CPU TEMPERATUR", get_cpu_temp),
        ("LAUFZEIT", get_uptime),
        ("SPEICHER", get_disk_space),
        ("SYSTEMINFO", lambda: "Beenden?")
    ]
    
    current_page = 0
    
    try:
        while True:
            title, func = pages[current_page]
            val = func()
            
            lcd.clear()
            lcd.write_string(title[:16])
            lcd.cursor_pos = (1, 0)
            lcd.write_string(val[:16])
            
            start_wait = time.time()
            while (time.time() - start_wait) < 30.0:
                if btn_nein.is_pressed:
                    current_page = (current_page + 1) % len(pages)
                    time.sleep(0.3) 
                    break 

                if btn_ja.is_pressed:
                    if current_page == len(pages) - 1:
                        lcd.clear()
                        lcd.write_string("ZURUECK...")
                        btn_ja.close()
                        btn_nein.close()
                        os.execv(sys.executable, ['python3', os.path.join(config.BASE_DIR, "main_menu.py")])
                    time.sleep(0.3)

                if btn_ja.is_pressed and btn_nein.is_pressed:
                    hold_start = time.time()
                    while btn_ja.is_pressed and btn_nein.is_pressed:
                        if (time.time() - hold_start) >= 2.0:
                            btn_ja.close()
                            btn_nein.close()
                            os.execv(sys.executable, ['python3', os.path.join(config.BASE_DIR, "main_menu.py")])
                        time.sleep(0.1)
                time.sleep(0.1)
                
    except KeyboardInterrupt:
        pass
    finally:
        if lcd: lcd.clear()
        btn_ja.close()
        btn_nein.close()

if __name__ == "__main__":
    main()